
#include <sys/types.h>

#include "stdio.h"
#include "error.h"
#include "floppy.h"
#include "standard.h"
#include "crc.h"

/************************************************************************
 * NAME:	floppy_init()
 *
 * DESCR:	Initialize a floppy structure to known values.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
floppy_init(struct floppy *floppy)
{
    floppy->write_protect = 0x00;
    floppy->tracks = 0;
    floppy->sides = 1;
    floppy->secsize = 256;			/* corresponds to 256 */
    floppy->encoding = UNKNOWN_ENCODING;
    floppy->sectors = 0;
    floppy->side[0] = NULL;
    floppy->side[1] = NULL;
}

/************************************************************************
 * NAME:	floppy_encoding_name()
 *
 * DESCR:	Returns a pointer to an abbreviation for the incoming
 *		sector/floppy encoding.
 *
 * ARGS:	if "abbrev" is TRUE, return an abbreviated version.
 *
 * RETURNS:	
 *
 * NOTES:	- these are the internal formats supported by the SVD
 *		  hardware...so they have to be explicitely listed.
 ************************************************************************/
char *
floppy_encoding_name(encoding encoding, int abbrev)
{
    if (abbrev) {
	switch(encoding) {
            case WD_FM:		return("SD");
            case WD_MFM:	return("DD");
            case H17_HSFM:	return("H17");
	    case AGCR6x2:	return("GCR6x2");
	    case AGCR5x3:	return("GCR5x3");
	    case RNIB:		return("RNIB");
            case RX02:		return("RXO2");
            case BLANK_SECTOR:  return("blank");
            default:		return("UNKNOWN");
	}
    } else {
	switch(encoding) {
            case WD_FM:		return("WD Single Density (FM)");
            case WD_MFM:	return("WD Double Density (MFM)");
            case H17_HSFM:	return("H17 Hard Sector Single Density (FM)");
	    case AGCR6x2:	return("Apple GCR 6x2");
	    case AGCR5x3:	return("Apple GCR 5x3");
	    case RNIB:		return("Apple Raw Nibble");
            case RX02:		return("DEC RXO2");
            case BLANK_SECTOR:  return("blank sector");
            default:		return("Unknown sector encoding");
	}
    }
}
	
/************************************************************************
 * NAME:	floppy_print_encoding()
 *
 * DESCR:	Print out the encoding for the high-level floppy structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
floppy_print_encoding(struct floppy *floppy, char *buffer)
{
    char	*ptr = buffer;

    ptr += sprintf(ptr,"Encoding: ");
    ptr += sprintf(ptr,"%s",floppy_encoding_name(floppy->encoding,FALSE));
    ptr += sprintf(ptr,"\n");

    return((int)(ptr-buffer));
}

/************************************************************************
 * NAME:	floppy_print()
 *
 * DESCR:	"print" out a floppy structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
floppy_print(struct floppy *floppy, char *buffer)
{
	
    buffer += sprintf(buffer,"Floppy Dump\n");
    buffer += sprintf(buffer,"---------------\n");

    buffer += sprintf(buffer,"Write protect: %s\n", (floppy->write_protect)?"ON":"off");
    buffer += sprintf(buffer,"Number of Tracks: %d\n", floppy->tracks);
    buffer += sprintf(buffer,"Number of Sides: %d\n", floppy->sides);
    buffer += floppy_print_encoding(floppy,buffer);
    buffer += sprintf(buffer,"Number of Sectors: %d\n", floppy->sectors);
    buffer += sprintf(buffer,"\n");
}

/************************************************************************
 * NAME:	floppy_sectormap()
 *
 * DESCR:	Fill the sector map for all of the sectors.
 *
 * ARGS:	
 *
 * RETURNS:	Nada.
 *
 * NOTES:	
 ************************************************************************/
void
floppy_sectormap(struct floppy *floppy)
{	
    int	track, side, sector;

    for (track=0; track < floppy->tracks; track++) {

	for (side=0; side < floppy->sides; side++) {

	    /* first, clear out the sectormap	*/
	    for (sector=0; sector < 256; sector++) {
		floppy->side[side][track].sectormap[sector] = SECTORMAP_UNDEFINED;
	    }

	    /* now load it */
	    for (sector=0; sector < floppy->side[side][track].sectors; sector++ ) {
		floppy->side[side][track].sectormap[floppy->side[side][track].sector[sector].sector] = sector;
	    }
	}
    }
}

/************************************************************************
 * NAME:	floppy_crc_set()
 *
 * DESCR:	Sets the CRCs.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
floppy_crc_set(struct floppy *floppy)
{
    return(floppy_crc_check(floppy,0,0,NULL));
}

/************************************************************************
 * NAME:	floppy_crc_check()
 *
 * DESCR:	Check or set the CRC codes for both the header and the
 *		data in the floppy structure.
 *
 * ARGS:	If "check" is non-zero, then only check, don't set.
 *
 * RETURNS:	0 if no errors, 1 upon header error, 2 upon body error,
 *		or 3 if both.
 *
 * NOTES:
 ************************************************************************/
int floppy_crc_check(struct floppy *floppy, int check, int verbose, char *prog)
{
    int				side;
    int				track;
    int				sector;
    int				value;
    int				retvalue = 0;

    for (track = 0; track < floppy->tracks; track++) {

	for (side = 0; side < floppy->sides; side++) {

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {

#define THIS_SECTOR	floppy->side[side][track].sector[sector]

		value = 0;

		switch(THIS_SECTOR.encoding) {

		    case WD_FM:
		    case WD_MFM:
			if (WD_crc_header(&(THIS_SECTOR),check)) {
			    /* error in header	*/
			    value |= 0x01;
			}
			if (WD_crc_body(&(THIS_SECTOR),check)) {
			    /* error in body	*/
			    value |= 0x02;
			}
			break;

		    case H17_HSFM:
			if (h17_crc_header(&(THIS_SECTOR),check)) {
			    value |= 0x01;
			}
			if (h17_crc_body(&(THIS_SECTOR),check)) {
			    value |= 0x02;
			}
			break;

		     case AGCR6x2:
		     case AGCR5x3:
			 break;


		    case RX02:	/* nothing to be done for this  yet	*/
			break;

		    case BLANK_SECTOR:
		    default:	/* nothing ever to be done for these 	*/
			break;
		}

		if (value != 0 && verbose) {
		    fprintf(stderr,"%s: %s CRC error track %d, side %d, sector %d (logical %d)\n",
			    prog, (value & 0x01)?"header":"data", track, side, sector, THIS_SECTOR.sector);
		}
		retvalue |= value;
	    }
	}
    }
    return(retvalue);
}

/************************************************************************
 * NAME:	floppy_free()
 *
 * DESCR:	Free the allocated members of the floppy structure.
 *		The floppy structure isn't freed itself.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
floppy_free(struct floppy *floppy)
{
    int	track, side, sector;

    /* free sector storage	*/

    for (side = 0; side < floppy->sides; side++) {
	for (track = 0; track < floppy->tracks; track++) {
	    free((void *)floppy->side[side][track].sector);
	}
	free((void *)floppy->side[side]);
    }
}

/************************************************************************
 * NAME:	floppy_overall_encoding()
 *
 * DESCR:	Figures out the overall encoding for a given floppy.
 *		This is done by looking at all of the sectors and finding
 *		out which format is in the majority.
 *
 * ARGS:	"mixed" is given as a pointer, and is filled in with TRUE
 *		if the floppy (or track) has mixed formats, FALSE otherwise.
 *		if "track" is given as something other than -1, then
 *		figure out the majority encoding for that track only.
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
encoding
floppy_overall_encoding(struct floppy *floppy, int *mixed, int track)
{
    int		i;
    int		side, sector;
    int		encoding_types[UNKNOWN_ENCODING];
    encoding	majority_encoding = UNKNOWN_ENCODING;
    int		majority_encoding_count = 0;
    int		track_limit;

    *mixed = FALSE;

    for (i=0; i < UNKNOWN_ENCODING; i++) {
	encoding_types[i] = 0;
    }

    if (track != -1) {
	track_limit = track + 1;
    } else {
	track = 0;
	track_limit = floppy->tracks;
    }

    for (; track < track_limit; track++) {
	for (side=0; side < floppy->sides; side++) {
	    for (sector=0; sector < floppy->side[side][track].sectors; sector++) {
		encoding_types[floppy->side[side][track].sector[sector].encoding] += 1;
	    }
	}
    }

    for (i=0; i < UNKNOWN_ENCODING; i++) {
	if (encoding_types[i] > majority_encoding_count) {
	    if (majority_encoding_count != 0) {
		*mixed = TRUE;
	    }
	    majority_encoding = i;
	    majority_encoding_count = encoding_types[i];
	}
    }

    return(majority_encoding);
}

/************************************************************************
 * NAME:	floppy_report()
 *
 * DESCR:	Generate basic floppy report.  Assumed to be called from
 *		something else.  Assumes that the floppy struct is filled.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	"machine readable" form outputs a set of text that is
 *		easily parsed and in the form:
 *
 *		 format,wp,sides,tracks,sectors,sectorexcep,encoding,encodingexcep
 *
 *		such as:
 *
 *		DMK,1,1,40,11,1,0,1
 *
 *	Which means DMK format, write-protected, 1 side, 40 tracks, 11 sectos
 *		with some "exceptions", single density with some exceptions
 ************************************************************************/
void
floppy_report(struct floppy *floppy, int level, int errors)
{
    int		i;
    int		track, side, sector;
    int		odd_sectors = FALSE;
    int		mixed_encoding;
    encoding	floppy_encoding;

    int		fake_sides = 0;

    if (level == 2 || level == 3 || level == 4) {	/* second level, show also disk params	*/

	/* check for the condition where some tracks have diff # sects	*/

	for (track=0; track < floppy->tracks; track++) {
	    for (side=0; side < floppy->sides; side++) {
		if (floppy->sectors != floppy->side[side][track].sectors) {
		    odd_sectors = TRUE;
		}
	    }
	}

	/* check for fake side labels	*/

	for (track=0; track < floppy->tracks; track++) {
	    for (side=0; side < floppy->sides; side++) {
		for (sector=0; sector < floppy->side[side][track].sectors; sector++) {
		    if (floppy->side[side][track].sector[sector].side > floppy->sides-1) {
			fake_sides++;
			break;		/* might as well break	*/
		    }
		}
	    }
	}

	/* check floppy encoding and mixed encoding	*/

	floppy_encoding = floppy_overall_encoding(floppy,&mixed_encoding,-1);

	if (level == 2 || level == 3) {
	    printf("%s %s%d tracks, %d sectors%s, %d side%s%s, %s encoding%s",
		   (floppy->write_protect)?"*":"",
		   (errors)?"(ERRORS) ":"",
		   floppy->tracks,
		   floppy->sectors, (odd_sectors)?"*":"",
		   floppy->sides, (floppy->sides==1)?"":"s", (fake_sides > 0)?"*":"",
		   floppy_encoding_name(floppy_encoding,TRUE), (mixed_encoding)?"*":"");
	    printf("\n");
	}

	if (level == 4) {
	    printf(",%d,%d,%d,%d,%d,%d,%d\n",
		   (floppy->write_protect)?1:0,
		   floppy->sides,
		   floppy->tracks,
		   floppy->sectors, (odd_sectors)?1:0,
		   floppy_encoding, (mixed_encoding)?1:0);
	}
    }

    if (level == 3) {		/* third level, report on structure	*/
	/* go thru each side, track, sector reporting weirdnesses	*/
	int	sector;
	int	track;
	int	side;

	for (side = 0; side < floppy->sides; side++) {
	    for (track = 0; track < floppy->tracks; track++) {
		for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {
		    printf("\ttrack %2d, side %2d, sector(%s) %2d, dam %x\n",
			   floppy->side[side][track].sector[sector].id,
			   floppy->side[side][track].sector[sector].side,
			   floppy_encoding_name(floppy->side[side][track].sector[sector].encoding,TRUE),
			   floppy->side[side][track].sector[sector].sector,
			   floppy->side[side][track].sector[sector].mark);
		}
	    }
	}
    }
}

/************************************************************************
 * NAME:	floppy_find_sector()
 *
 * DESCR:	Given a logical track number (which is assumed to be valid)
 *		and a potential logical sector number, figure out if it is
 *		a VALID sector number.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the given logical sector was found on the track.
 *
 * NOTES:	
 ************************************************************************/
int
floppy_find_sector(struct floppy *floppy, int ltrack, int lsector)
{

}

/************************************************************************
 * NAME:	floppy_find_track()
 *
 * DESCR:	Check to see if the given logical track exists.  Normally,
 *		the physical track is the same as the logical track.  Just
 *		FYI.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the given logical track was found.
 *
 * NOTES:	
 ************************************************************************/
int
floppy_find_track(struct floppy *floppy, int ltrack)
{
}

/************************************************************************
 * NAME:	floppy_copy_track()
 *
 * DESCR:	Copies a source track to a destination track.  The
 *		original contents of destination are destroyed.  The
 *		original data in the location of source is unchanged.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- a BIG assumption is made that the destination track
 *		  is EXACTLY the same size as the source track.
 ************************************************************************/
void
floppy_copy_track(struct floppy *floppy, int source, int destination)
{
    int	side, sector;
    int	i;

    for (side=0; side < floppy->sides; side++) {
	for (i=0; i < 256; i++) {
	    floppy->side[side][destination].sectormap[i] = floppy->side[side][source].sectormap[i];
	}
	for (sector = 0; sector < floppy->side[side][source].sectors && 
		         sector < floppy->side[side][destination].sectors ; sector++) {
	    floppy->side[side][destination].sector[sector].id = floppy->side[side][source].sector[sector].id;
	    floppy->side[side][destination].sector[sector].side = floppy->side[side][source].sector[sector].side;
	    floppy->side[side][destination].sector[sector].sector = floppy->side[side][source].sector[sector].sector;
	    floppy->side[side][destination].sector[sector].sizecode = floppy->side[side][source].sector[sector].sizecode;
	    floppy->side[side][destination].sector[sector].size = floppy->side[side][source].sector[sector].size;
	    floppy->side[side][destination].sector[sector].mark = floppy->side[side][source].sector[sector].mark;
	    floppy->side[side][destination].sector[sector].headCRC = floppy->side[side][source].sector[sector].headCRC;
	    floppy->side[side][destination].sector[sector].dataCRC = floppy->side[side][source].sector[sector].dataCRC;
	    floppy->side[side][destination].sector[sector].encoding = floppy->side[side][source].sector[sector].encoding;
	    memcpy(floppy->side[side][destination].sector[sector].data,
		   floppy->side[side][source].sector[sector].data,
		   MIN(floppy->side[side][source].sector[sector].size,floppy->side[side][destination].sector[sector].size));
	}
    }
}

/************************************************************************
 * NAME:	floppy_dump_sector()
 *
 * DESCR:	Dump the given sector/track in a nice readable format
 *		on the given file descriptor.
 *
 *		The given format indicates how much info to generate.
 *
 * ARGS:	track - either the logical or physical track number
 *		sector - either the logical or physical sector number
 *		phys - TRUE if the sector/track is physical (not logical)
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
floppy_dump_sector(FILE *fd,struct floppy *floppy, int track, int sector, 
		   int phys, int verbosity)
{
    unsigned char	*data;
    int			 i,j;
    int			 ptrack, psector;
    int			 ltrack, lsector;
    int			 mark;

/* BIG NOTE - the following LOGICAL part is broken - it doesn't find the
 *	real LOGICAL track - it assumes that the logical track is the
 *	same as the physical track.
 */

    if (phys) {
	ptrack = track;
	psector = sector;
	ltrack = floppy->side[0][ptrack].sector[psector].id;
	lsector = floppy->side[0][ptrack].sector[psector].sector;
    } else {
	ltrack = track;
	lsector = sector;
	ptrack = ltrack;	/* broken here - see note above */
	psector = floppy->side[0][ptrack].sectormap[lsector];
    }

    data = floppy->side[0][ptrack].sector[psector].data;
    mark = floppy->side[0][ptrack].sector[psector].mark;

    /* now format the data - do 16x16 with a character display on side	*/

    fprintf(fd,"Phys Track: %d, Phys Sector: %d (logical %d, %d), Mark %02x\n",
	                                                            ptrack,psector,
	                                                            ltrack,lsector,
	                                                            mark);
    for (i=0; i < 16; i++) {
	if (verbosity == 1) {
	    fprintf(fd,"%03o: ",i*16);
	} else {
	    fprintf(fd,"%02x: ",i*16);
	}
	for (j=0; j < 16; j++) {
	    if (verbosity == 1) {
		fprintf(fd,"%03o ", data[i*16+j]);
	    } else {
		fprintf(fd,"%02x ", data[i*16+j]);
	    }
	}
	fprintf(fd," - ");
	for (j=0; j < 16; j++) {
	    fprintf(fd,"%c",(data[i*16+j] >= ' ' && data[i*16+j] <= 'z')?data[i*16+j]:'.');
	}
	fprintf(fd,"\n");
    }



    /* used to dump an apple 2 sector for debugging	*/

    {
#ifdef NOTDEF
	unsigned char	agcrbuf[342];

	unsigned char vol[2];
	unsigned char track[2];
	unsigned char sector[2];
	unsigned char cs[2];

	a2_gcr4x4_map(floppy->side[0][ptrack].sector[psector].id,&track[0],&track[1]);
	a2_gcr4x4_map(floppy->side[0][ptrack].sector[psector].volume,&vol[0],&vol[1]);
	a2_gcr4x4_map(floppy->side[0][ptrack].sector[psector].sector,&sector[0],&sector[1]);
	a2_gcr4x4_map(floppy->side[0][ptrack].sector[psector].hdrchecksum,&cs[0],&cs[1]);

	fprintf(fd,"(%02x %02x %02x - %02x %02x %02x %02x %02x %02x %02x %02x - %02x %02x %02x - %02x %02x %02x - %02x - %02x %02x %02x)\n",
		floppy->side[0][ptrack].sector[psector].hdrprolog[0],
		floppy->side[0][ptrack].sector[psector].hdrprolog[1],
		floppy->side[0][ptrack].sector[psector].hdrprolog[2],
		vol[0], vol[1],
		track[0], track[1],
		sector[0], sector[1],
		cs[0], cs[1],
		floppy->side[0][ptrack].sector[psector].hdrepilog[0],
		floppy->side[0][ptrack].sector[psector].hdrepilog[1],
		floppy->side[0][ptrack].sector[psector].hdrepilog[2],
		floppy->side[0][ptrack].sector[psector].dataprolog[0],
		floppy->side[0][ptrack].sector[psector].dataprolog[1],
		floppy->side[0][ptrack].sector[psector].dataprolog[2],
		floppy->side[0][ptrack].sector[psector].datachecksum,
		floppy->side[0][ptrack].sector[psector].hdrepilog[0],
		floppy->side[0][ptrack].sector[psector].hdrepilog[1],
		floppy->side[0][ptrack].sector[psector].hdrepilog[2]);

	a2_6x2_gcr_translate(floppy->side[0][ptrack].sector[psector].data,agcrbuf);

	for (i=0; i < 342; i++) {
	    fprintf(fd,"%02x, ",agcrbuf[i]);
	    if ((i+1)%16 == 0) {
		fprintf(fd,"\n");
	    }
	}
	fprintf(fd,"- %02x",floppy->side[0][ptrack].sector[psector].datachecksum);
	fprintf(fd,"\n");
#endif
    }


}

/************************************************************************
 * NAME:	floppy_find_dir_track()
 *
 * DESCR:	Finds the directory track and returns it.
 *		Returns the physical track number.
 *
 * ARGS:	startrack - the track to start with, used for looking
 *				further for a directory if a bad one found
 *
 * RETURNS:	physical track number, or max tracks if not found
 *
 * NOTES:	- uses the fact that sectors of the directory track are
 *		  marked differently than sectors on other tracks
 *		- the basic assumption is that there is only one directory
 *		  track on the diskette.  So this routine looks at at least
 *		  three tracks to find out what is the directory mark and
 *		  what is the normal mark.
 *		- checking is started at track zero
 *		- because of the weirdness of the TRSDOS23, all of the sector
 *		  marks on a track must be identical or the track is ignored
 *		  for this routine.
 *		- only side zero is checked
 ************************************************************************/
int
floppy_find_dir_track(struct floppy *floppy, int startrack)
{
    int		track;
    int		mark, mark1, mark2, mark3;

    /* get the first track that has a consistent mark	*/

    for (track = startrack; track < floppy->tracks; track++) {
	mark = floppy_get_track_mark(floppy,track);
	if (mark != 0) {
	    mark1 = mark;
	    break;
	}
    }

    /* get the second track that has a consistent mark	*/

    for (; track < floppy->tracks; track++) {
	mark = floppy_get_track_mark(floppy,track);
	if (mark != 0) {
	    mark2 = mark;
	    break;
	}
    }

    /* get the third track that has a consistent mark	*/

    for (; track < floppy->tracks; track++) {
	mark = floppy_get_track_mark(floppy,track);
	if (mark != 00) {
	    mark3 = mark;
	    break;
	}
    }

    if (mark1 != mark2) {	/* found directory in track 0 or 1	*/
	if (mark1 == mark3) {
	    return(1);
	} else {
	    return(0);
	}
    } else {			/* we have the normal mark, find directory */
	for (track=startrack+2; track < floppy->tracks; track++) {
	    mark3 = floppy_get_track_mark(floppy,track);
	    if( mark3 != 0 && mark3 != mark1 ) {
		return(track);
	    }
	}
    }

    return(floppy->tracks);	/* indicates that a directory wasn't found */
}

/************************************************************************
 * NAME:	floppy_get_track_mark()
 *
 * DESCR:	Gets the mark for the given physical track.
 *
 * ARGS:	
 *
 * RETURNS:	the track mark, or 0x00 if no consistent mark was found.
 *
 * NOTES:	
 ************************************************************************/
int
floppy_get_track_mark(struct floppy *floppy, int track)
{
    int	mark = 0;
    int	i;

    mark = floppy->side[0][track].sector[0].mark;

    for (i=1; i < floppy->sectors; i++) {
	if (mark != floppy->side[0][track].sector[i].mark) {
	    return(0);
	}
    }

    return(mark);
}

/************************************************************************
 * NAME:	floppy_read_sector()
 *
 * DESCR:	An interface directly into the floppy data, without
 *		knowing the floppy structure.  Also hides the difference
 *		between logical and physical.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the sector was read, FALSE otherwise.
 *
 * NOTES:	- buffer is assumed to be big enough for the sector
 ************************************************************************/
int
floppy_read_sector(struct floppy *floppy, int side, int track, int sector, char *buffer)
{
    if (side < floppy->sides) {
	if (track < floppy->tracks) {
	    if (sector < 256 && SectorMap(floppy,side,track,sector) < floppy->sectors) {
		memcpy(buffer,
		       LogicalSector(floppy,side,track,sector).data,
		       LogicalSector(floppy,side,track,sector).size);
		return(TRUE);
	    }
	}
    }

    return(FALSE);
}

/************************************************************************
 * NAME:	floppy_write_sector()
 *
 * DESCR:	An interface directly into the floppy data, without
 *		knowing the floppy structure.  Also hides the difference
 *		between logical and physical.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the sector was written, FALSE otherwise.
 *
 * NOTES:	- buffer is assumed to be big enough for the sector
 ************************************************************************/
int
floppy_write_sector(struct floppy *floppy, int side, int track, int sector, char *buffer)
{
    if (side < floppy->sides) {
	if (track < floppy->tracks) {
	    if (sector < 256 && SectorMap(floppy,side,track,sector) < floppy->sectors) {
		memcpy(LogicalSector(floppy,side,track,sector).data,
		       buffer,
		       LogicalSector(floppy,side,track,sector).size);
		return(TRUE);
	    }
	}
    }

    return(FALSE);
}

/************************************************************************
 * These routines implement the file extension registry.  This allows
 * file formats to be "guessed" more specifically given a file extension.
 * Note that the file formats are used ONLY to prioritize the guessor.
 * In other words, even if a file is of a given extension, it will still
 * need to pass the registered guess to qualify for the target format.
 * If this ain't good enough, then force the format!
 ************************************************************************/
#define FILE_EXTENSIONS	50

struct file_ext_item {
    char	*ext;		/* the target file extension		*/
    char	*id;		/* related file format to try to guess	*/
    int		 strength;	/* the strength of this entry		*/
};

static struct file_ext_item	floppy_fileext_registry[FILE_EXTENSIONS];
static int			floppy_fileext_count = 0;

/************************************************************************
 * NAME:	floppy_fileext_sort_fn()
 *
 * DESCR:	Function that can sort entries in the extension registry.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static int
floppy_fileext_sort_fn(struct file_ext_item *one, struct file_ext_item *two)
{
    if (one->strength < two->strength) {
	return(-1);
    }

    if (one->strength > two->strength) {
	return(1);
    }

    return(0);
}

/************************************************************************
 * NAME:	floppy_fileext_register()
 *
 * DESCR:	Register the given file extension as being something
 *		that should be guessed for the given format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- the file extension array is sorted every time...this
 *		  is way overkill...but shouldn't be much of a problem.
 *		- the guessor id isn't checked for validity
 ************************************************************************/
int
floppy_fileext_register(char *ext, int strength, char *id)
{
    if (floppy_fileext_count == FILE_EXTENSIONS) {
	return(FALSE);
    }

    floppy_fileext_registry[floppy_fileext_count].ext = ext;
    floppy_fileext_registry[floppy_fileext_count].strength = strength;
    floppy_fileext_registry[floppy_fileext_count].id = id;

    floppy_fileext_count++;

    qsort((void *)floppy_fileext_registry,
	  (size_t)floppy_fileext_count,
	  sizeof(struct file_ext_item),
	  floppy_fileext_sort_fn);

    return(TRUE);
}

/************************************************************************
 * NAME:	floppy_fileext_list_print()
 *
 * DESCR:	DEBUGGING - used to print the list of file extensions.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
floppy_fileext_list_print(void)
{
    int i;

    for (i=0; i < floppy_fileext_count; i++) {
	fprintf(stderr,"%d: %s, %s, %s specific\n",
		i,
		floppy_fileext_registry[i].ext,
		floppy_fileext_registry[i].id,
		(floppy_fileext_registry[i].strength==EXT_VERY_SPECIFIC)?"very":(floppy_fileext_registry[i].strength==EXT_SOMEWHAT_SPECIFIC)?"somewhat":"not");
    }
}


/************************************************************************
 * NAME:	floppy_fileext_list()
 *
 * DESCR:	Returns an array of format guessors to try for the given
 *		file extension.
 *
 * ARGS:	
 *
 * RETURNS:	the count of guessors for the given extension...may be zero
 *		the given pointer is filled in with the list
 *
 * NOTES:	- the caller needs to FREE this list!
 *		- the list is ORDERED, the guessors should be called in this order
 ************************************************************************/
static int
floppy_fileext_list(char *ext, char ***list)
{
    int	i;
    int	count = 0;

    for (i=0; i < floppy_fileext_count; i++) {
	if (strcasecmp(floppy_fileext_registry[i].ext,ext) == 0) {
	    count++;
	}
    }

    if (count == 0 || (*list = (char **)malloc(sizeof(char *)*count)) == NULL) {
	return(0);
    }

    count = 0;

    for (i=0; i < floppy_fileext_count; i++) {
	if (strcasecmp(floppy_fileext_registry[i].ext,ext) == 0) {
	    (*list)[count++] = floppy_fileext_registry[i].id;
	}
    }

    return(count);
}

/************************************************************************
 * NAME:	floppy_fileext_query()
 *
 * DESCR:	Will generate a report into the given buffer describing
 *		the current file extension support.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- verbose isn't used
 ************************************************************************/
void
floppy_fileext_query(char *buffer, int verbose)
{
    int		i, j, k;
    char	**list;

    for (i=0; i < floppy_fileext_count; i++) {
	for (j=i-1; j >=0; j--) {
	    if (strcasecmp(floppy_fileext_registry[i].ext,floppy_fileext_registry[j].ext) == 0) {
		break;
	    }
	}
	if (j == -1) {
	    j = floppy_fileext_list(floppy_fileext_registry[i].ext,&list);
	    if (j > 0) {
		buffer += sprintf(buffer,"%s --> ",floppy_fileext_registry[i].ext);
		for (k=0; k < j; k++) {
		    buffer += sprintf(buffer,"%s ", list[k]);
		}
		buffer += sprintf(buffer,"\n");
		free(list);
	    }
	}
    }
}

/************************************************************************
 * NAME:	floppy_file_basename()
 *
 * DESCR:	Returns a pointer to the "basename" of a given file.  The
 *		basename is defined as the text between a file path
 *		indicator ("/" or "\") and the end of the string.  If there is
 *		no base name (like for a zero filename or something
 *		that only has an extension) the a pointer the beginning
 *		of where the basename should be is returned.  This would
 *		therefore, be a pointer to '\0'.
 *
 * ARGS:	filename - \0 terminated string
 *
 * RETURNS:	pointer to the basename within the filename.  Will always
 *		return something, potentially a pointer to '\0'.
 *
 * NOTES:	- filename should NOT be NULL
 ************************************************************************/
char *
floppy_file_basename(char *filename)
{
    int	i;

    for (i=strlen(filename)-1; i >=0; i--) {
	if (filename[i] == '\\' || filename[i] == '/') {
	    break;			/* found where the base should start	*/
	}
    }

    return(&(filename[i+1]));
}


/************************************************************************
 * NAME:	floppy_file_extension()
 *
 * DESCR:	Returns a pointer into a filename, pointing to the extension
 *		for that filename.  It is assumed that the filename is
 *		terminated with \0.  Further, the pointer returned minus 1
 *		should always be '.'.
 *
 * ARGS:	filename - \0 terminated string
 *
 * RETURNS:	pointer to the extension within the filename.  If there
 *		is no extension, then NULL is returned (not '\0').
 *
 * NOTES:	- the pointer returned is NOT a copy of the original,
 *		  therefore it's contents have the same lifespan as the
 *		  original filename.
 *		- if the file ends in '.' then an extension is assumed,
 *		  thought it is \0.
 ************************************************************************/
char *
floppy_file_extension(char *filename)
{
    int	i;

    for (i=strlen(filename)-1; i >=0; i--) {
	if (filename[i] == '\\' || filename[i] == '/') {
	    break;		/* no extension apparently */
	}
	if (filename[i] == '.') {
	    return (&filename[i+1]);
	}
    }

    return(NULL);
}


/************************************************************************
 * The following routines implement the floppy_format registry for
 * guessing and dumping out of the formats.  Up to FLOPPY_FORMATS number
 * of floppy formats can be registers.
 ************************************************************************/
#define FLOPPY_FORMATS	20

struct floppy_format_item {
    char	*id;
    char	*description;
    int		(*guesser)(int);
    int		(*reader)(int,struct floppy *);
    int		(*dumper)(int,struct floppy *);
    int		(*reporter)(int,struct floppy *,int);
};

static struct floppy_format_item	floppy_format_registry[FLOPPY_FORMATS];
static int				floppy_format_count = 0;

typedef void (*fnptr)(void);
extern fnptr floppy_format_init_list[];

/************************************************************************
 * NAME:	floppy_format_init()
 *
 * DESCR:	Initialize all of the floppy formats that have been
 *		described in the make file.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- floppy_format_init_list is defined in the Makefile.
 ************************************************************************/
void
floppy_format_init()
{
    int	i = 0;

    for (;;i++) {

	if( floppy_format_init_list[i] == NULL ) {
	    break;
	}

	(*(floppy_format_init_list[i]))();
    }
}

/************************************************************************
 * NAME:	floppy_format_register()
 *
 * DESCR:	Registers a floppy input format.  The order of registration
 *		is important, because that is the order of guessing.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:
 *		- the "id" is passed as a character pointer and is stored
 *		  as such...it is NOT COPIED.  So the caller should be
 *		  sure to pass a pointer to something that won't change
 *		  (like a constant).
 ************************************************************************/
int
floppy_format_register(char *id,
		       char *description,
		       int (*guesser)(int),
		       int (*reader)(int,struct floppy *),
		       int (*dumper)(int,struct floppy *),
		       int (*reporter)(int,struct floppy *,int))
{
    if (floppy_format_count == FLOPPY_FORMATS) {
	return(FALSE);
    }
    
    floppy_format_registry[floppy_format_count].id = id;
    floppy_format_registry[floppy_format_count].description = description;
    floppy_format_registry[floppy_format_count].guesser = guesser;
    floppy_format_registry[floppy_format_count].reader = reader; 
    floppy_format_registry[floppy_format_count].dumper = dumper;
    floppy_format_registry[floppy_format_count].reporter = reporter;
    floppy_format_count++;

    return(TRUE);
}

/************************************************************************
 * NAME:	floppy_format_lookup()
 *
 * DESCR:	Returns a pointer to a registry entry based upon the given
 *		format id.  If there is no entry for the ID, NULL is returned.
 *
 * ARGS:	
 *
 * RETURNS:	pointer to the associated format registry entry, or NULL if
 *		it doesn't exist.
 *
 * NOTES:	
 ************************************************************************/
static struct floppy_format_item *
floppy_format_lookup(char *format_id)
{
    int	i;

    for (i=0; i < floppy_format_count; i++) {
	if (floppy_format_registry[i].id == format_id) {
	    return(&(floppy_format_registry[i]));
	}
    }

    return ((struct floppy_format_item *)NULL);
}

/************************************************************************
 * NAME:	floppy_format_guesser()
 *
 * DESCR:	Attempts to guess the format of the incoming file.
 *
 * ARGS:	
 *
 * RETURNS:	returns the ID (a char string) of the format if found.
 *		A NULL is returned if it is not found.
 *
 * NOTES:	
 ************************************************************************/
char *
floppy_format_guesser(int fd, char *ext)
{
    int				  i;
    char			**formatlist;
    int				  formats;
    struct floppy_format_item	 *format;

    if (ext != NULL) {

	if ((formats = floppy_fileext_list(ext,&formatlist)) != 0) {

	    for (i=0; i < formats; i++) {
#ifdef NOTDEF
		M1("calling out to guessor %s\n",formatlist[i]);
#endif
		format = floppy_format_lookup(formatlist[i]);
		if (format != NULL && format->guesser != NULL) {
		    if ((format->guesser)(fd)) {
			free(formatlist);
			return(format->id);
		    }
		}
		lseek(fd,(off_t)0,SEEK_SET);
	    }
	    free(formatlist);
	}

    } else {


	for (i=0; i < floppy_format_count; i++) {
	    format = &floppy_format_registry[i]; 
	    if ( format->guesser != NULL && (format->guesser)(fd)) {
		return(format->id);
	    }
	    lseek(fd,(off_t)0,SEEK_SET);
	}

    }

    return(NULL);
}

/************************************************************************
 * NAME:	floppy_format_reader()
 *
 * DESCR:	Given a floppy_format (which was probably found by guessing
 *		or may have been selected by command line options) call
 *		the reader routine.
 *
 * ARGS:	
 *
 * RETURNS:	the return value of the reading routine is sent back.
 *		and undefined return value comes back if the format is bad
 *
 * NOTES:	
 ************************************************************************/
int
floppy_format_reader(char *format_id, int fd, struct floppy *floppy)
{
    int	i;

    for (i=0; i < floppy_format_count; i++) {
	if (floppy_format_registry[i].id == format_id) {
	    return((*floppy_format_registry[i].reader)(fd,floppy));
	}
    }
    return(0);		/* should pick a better return value here	*/
}

/************************************************************************
 * NAME:	floppy_format_dumper()
 *
 * DESCR:	Given a floppy_format (which was probably found by guessing
 *		or may have been selected by command line options) call
 *		the dumper routine.
 *
 * ARGS:	
 *
 * RETURNS:	the return value of the writing routine is sent back.
 *		and undefined return value comes back if the format is bad
 *
 * NOTES:	
 ************************************************************************/
int
floppy_format_dumper(char *format_id, int fd, struct floppy *floppy)
{
    int	i;

    for (i=0; i < floppy_format_count; i++) {
	if (floppy_format_registry[i].id == format_id) {
	    return((*floppy_format_registry[i].dumper)(fd,floppy));
	}
    }
    return(0);		/* should pick a better return value here	*/
}

/************************************************************************
 * NAME:	floppy_format_report()
 *
 * DESCR:	Given a floppy_format (which was probably found by guessing
 *		or may have been selected by command line options) call
 *		the report routine.
 *
 * ARGS:	
 *
 * RETURNS:	the return value of the reading routine is sent back.
 *		and undefined return value comes back if the format is bad
 *
 * NOTES:	
 ************************************************************************/
int
floppy_format_report(char *format_id, int fd, struct floppy *floppy, int level)
{
    int	i;

    for (i=0; i < floppy_format_count; i++) {
	if (floppy_format_registry[i].id == format_id) {
	    return((*floppy_format_registry[i].reporter)(fd,floppy,level));
	}
    }
    return(0);		/* should pick a better return value here	*/
}

/************************************************************************
 * NAME:	floppy_format_list_string()
 *
 * DESCR:	Generates a string listing all of the possible floppy
 *		formats for reading/writing.  This string will probably
 *		be used to show a command line "Usage" message.
 *
 * ARGS:	- lists output formats if output is TRUE (input otherwise)
 *		- will do a multi-line listing if verbose is TRUE
 *
 * RETURNS:
 *
 * NOTES:
 ************************************************************************/
char *
floppy_format_list_string(char *buffer, int output, int verbose)
{
    int		 first = TRUE;
    char	*ptr = buffer;
    int		 i;

    for (i=0; i < floppy_format_count; i++) {
	if (!output || floppy_format_registry[i].dumper != NULL) {
	    if (verbose) {
		ptr += sprintf(ptr,"%s\t%s\n", 
				         floppy_format_registry[i].id,
			                 floppy_format_registry[i].description);
	    } else {
		if (first) {
		    first = FALSE;
		} else {
		    ptr += sprintf(ptr,", ");
		}
		ptr += sprintf(ptr,"%s",floppy_format_registry[i].id);
	    }
	}
    }

    return(buffer);
}

/************************************************************************
 * NAME:	floppy_format_list_select()
 *
 * DESCR:	Find and return the id of the given floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	the given id if found, NULL if no matching format.
 *
 * NOTES:	- the selection is done via a string.  This is the ONLY
 *		  time that the floppy format id is treated as a string,
 *		  in all other cases it is just an ID number that is
 *		  handed around.
 *		- this routine assumes that the set of output formats is
 *		  a subset of the input formats.
 ************************************************************************/
char *
floppy_format_list_select(char *selection, int output)
{
    int	    i;
    char   *ptr;
    size_t  count;

    /* first find the dash in the selection if there is one	*/

    for ( ptr = selection; *ptr; ptr++) {
	if (*ptr == '-') {
	    break;
	}
    }

    count = ptr - selection;

    /* only compare up to the dash	*/

    for (i=0; i < floppy_format_count; i++) {
	if (!output || floppy_format_registry[i].dumper != NULL) {
	    if (strncasecmp(floppy_format_registry[i].id,selection,count) == 0) {
		return(floppy_format_registry[i].id);
	    }
	}
    }

    return(NULL);
}

/************************************************************************
 * NAME:	floppy_format_add_params()
 *
 * DESCR:	Returns a pointer to a list of additional paramters that
 *		were specified for the given format.  NULL is returned
 *		(as opposed to a pointer to a \0) if there are no
 *		additional parameters.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- assumes that the incoming selection is \0  terminated.
 ************************************************************************/
char *
floppy_format_add_params(char *selection)
{
    int	  i;
    char *ptr;

    for ( ptr = selection; *ptr; ptr++) {
	if (*ptr == '-') {
	    return(++ptr);
	}
    }

    return(NULL);
}

/************************************************************************
 * NAME:	floppy_sector_swap()
 *
 * DESCR:	Swaps two physical sectors.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
floppy_sector_swap(struct track *track, int a, int b)
{
    struct sector	temp;

    if (a != b) {

	/* re-do sectormap */

	track->sectormap[track->sector[a].sector] = b;
	track->sectormap[track->sector[b].sector] = a;

	/* then swap the sectors */

	temp = track->sector[a];
	track->sector[a] = track->sector[b];
	track->sector[b] = temp;
    }
}

/************************************************************************
 * NAME:	floppy_find_lowest_sector()
 *
 * DESCR:	Finds the lowest numbered sector.
 *
 * ARGS:	
 *
 * RETURNS:	The logical sector number of the lowest logical sector.
 *
 * NOTES:	
 ************************************************************************/
int
floppy_find_lowest_sector(struct track *track)
{
    int	i;

    for (i=0; i < 256; i++) {
	if (track->sectormap[i] != SECTORMAP_UNDEFINED) {
	    return(i);
	}
    }

    return(SECTORMAP_UNDEFINED);
}


/************************************************************************
 * NAME:	floppy_interleave()
 *
 * DESCR:	Given a loaded floppy, generate a physical interleave
 *		for it.
 *
 *		This routine is defined to do the following:
 *		  - It operates on each track independently
 *		  - the lowest numbered sector is placed physically first
 *		  - sectors are physically interleaved from there
 *			- the sector closest to the last sector + I is next
 *			- if (sector+I) has been generated, then do (sector+I+1)
 *			-   (repeat the above +1 until a sector is generated)
 *
 * ARGS:	
 *
 * RETURNS:	- if any errors occur, no interleave is done
 *
 * NOTES:	- pointers to sectors are swizzled within the array
 *		- this routine only works with images with contiguous
 *		  sectors - disqualifying copy-protected images
 ************************************************************************/
void
floppy_interleave(struct floppy *floppy, int interleave)
{
#define MAXSECTORS	80

    int	track;
    int	side;
    int	i;
    int	first_sector;
    int this_track_sectors;
    int	offset;

    int	used_sectormap[MAXSECTORS];

    for (track=0; track < floppy->tracks; track++) {

	for (side=0; side < floppy->sides; side++) {

	    /* first, clear out the used sectormap	*/

	    this_track_sectors = floppy->side[side][track].sectors;

	    for (i=0; i < this_track_sectors; i++) {
		used_sectormap[i] = FALSE;
	    }

	    /* find the first sector	*/

	    first_sector = floppy_find_lowest_sector(&floppy->side[side][track]);

	    /* check that sectors are contiguous from the first	*/

	    for (i=first_sector; i < first_sector+this_track_sectors; i++) {
		if (floppy->side[side][track].sectormap[i] == SECTORMAP_UNDEFINED) {
		    return;
		}
	    }

	    /* now conduct the interleave...starting with the first sector	*/

	    offset = 0;
	    for (i=0; i < this_track_sectors-1; i++) {

		floppy_sector_swap(&floppy->side[side][track],i,floppy->side[side][track].sectormap[offset+first_sector]);
		used_sectormap[offset] = TRUE;

		/* do the interleave as requested, bump up by one if necessary	*/

		offset = (offset + interleave) % this_track_sectors;
		while (used_sectormap[offset]) {
		    offset = (offset + 1) % this_track_sectors;
		}
	    }

	    /* at this point the last sector is already in the right place	*/
	}
    }
}
